# -*- coding: utf-8 -*-
"""
Created on Thu Mar 27 17:42:23 2025

@author: semik
"""

# %%
# Import standard packages
from numpy import linspace
from scipy.stats.kde import gaussian_kde
from numpy.random import default_rng
import copy
import math
import numpy as np
import matplotlib.pyplot as plt
import sequence_jacobian as sj
from copy import deepcopy
import scipy
import pandas

# %%State-dependent Phillips Curve 

# In[2]:
calibration = {'eis': 1,     # EIS
               'rho_e': 0.98,  # Persistence of idiosyncratic productivity shocks
               'sd_e': 0.92,   # Standard deviation of idiosyncratic productivity shocks
               'rho_p': 0.98,  #  Persistence of idiosyncratic productivity shocks
               'sd_p': 0.92,   # Standard deviation of idiosyncratic productivity shocks
               'yss': 1,        # Output steady state
               'Y': 1,        # Output
               'X': 1,        
               'r_ante': 0.005, # target real interest rate
               'min_a':-1,    # Minimum asset level on the grid
               'max_a': 1000,   # Maximum asset level on the grid
               'n_a': 500,     # Number of asset grid points
               'n_e': 11,       # Number of productivity grid points
               'yshock':0,
               'curve':0.5,
               'phi':207.6748604782035,                
               'epsilon':11,
               'G': 0.2,  # Government spending
               'B': 2.8,  # Government debt
               }    


print('Calibration done')


# In[3]: 

# initialize
def hh_init(a_grid, z, r, eis,dfshock,p):
    coh = (1 + r) * a_grid[np.newaxis, :] + z[:, np.newaxis]  + p[:, np.newaxis] 
    Va = (1 + r) * (coh) ** (-1 / eis)
    return Va

# backward step
@sj.het(exogenous='Pi',  # <-- this means our transition matrix will be fed into the model as Pi (use this for forward iteration)
        policy='a',  # <-- this means our endogenous state variable is a, defined over grid a_grid (we use this to check convergence)
        backward='Va',  # <-- this means we're iterating over variable Va, whose future value is Va_p (solver needs to know this to iterate!)
        backward_init=hh_init)
def hh(Va_p, a_grid, z, r, beta, eis,dfshock,Y,p):
    uc_nextgrid = beta *2.72**(dfshock)* Va_p  # u'(c') on tomorrow's grid
    c_nextgrid = uc_nextgrid ** (-eis) # c' on tomorrow's grid
    coh = (1 + r) * a_grid[np.newaxis, :] + z[:, np.newaxis] + p[:, np.newaxis] # cash on hand on today's grid
    a = sj.interpolate.interpolate_y(c_nextgrid + a_grid, coh, a_grid)  # this plots (c_next + a', a') pairs and computes policy a' from interpolation on coh
    sj.misc.setmin(a, a_grid[0])  # impose borrowing constraint
    c = coh - a  # back out consumption
    Va = (1 + r) * (c) ** (-1 / eis)  # V'(a)
    return Va, a, c,coh
    

# forward iteration is done automatically!

def income_cyclical(Y,Z,P,T,e_grid, e_pdf, zeta,zetap,markup_ss,p_grid,p_pdf):
    N =Y
    gamma_N = e_grid ** (zeta * np.log(N)) / np.vdot(e_grid ** (1 + zeta * np.log(N)), e_pdf)
    z = Z * e_grid *gamma_N
    pretax = (Y/markup_ss) * e_grid *gamma_N
    #gamma_P = p_grid ** (zetap * np.log(P/((1-T)*(1-1/markup_ss)))) / np.vdot(p_grid ** (1 + zetap * np.log(P/((1-T)*(1-1/markup_ss)))), p_pdf)
    gamma_P = p_grid ** (zetap * np.log(Y)) / np.vdot(p_grid ** (1 + zetap * np.log(Y)), p_pdf)
    p = (P)* p_grid * gamma_P
    p_pretax = (Y*(1-1/markup_ss))* p_grid * gamma_P
    return z, p, pretax, p_pretax


def make_grids_pdf(rho_e, rho_p, sd_e, sd_p, n_e, min_a, max_a, n_a):
    e_grid, e_pdf, Pi = sj.grids.markov_rouwenhorst(rho_e, sd_e, n_e)
    p_grid, p_pdf, Pi = sj.grids.markov_rouwenhorst(rho_p, sd_p, n_e)
    a_grid = sj.grids.asset_grid(min_a, max_a, n_a)
    return e_grid, e_pdf, Pi, a_grid,  p_grid, p_pdf

household_cyc = hh.add_hetinputs([make_grids_pdf, income_cyclical])

@sj.simple
def ex_post_rate(r_ante):
    r = r_ante 
    return r

@sj.simple
def fiscal(B, r, G, Y, Tr,dfshock,markup_ss):
    T = ((1 + r) * B(-1) + G - B + Tr)/(Y) # total tax burden
    Z = (Y/markup_ss)*(1 - T)  # after tax income
    P = Y*(1-1/markup_ss)*(1-T)
    deficit = G + Tr - T
    return T, Z, deficit,P

@sj.simple
def mkt_clearing_simple(A, B, Y, C, G):
    asset_mkt = A - B
    goods_mkt = Y - C - G 
    return asset_mkt, goods_mkt

ha_simple = sj.create_model([household_cyc, ex_post_rate, mkt_clearing_simple])
ha_cyc = sj.create_model([household_cyc, ex_post_rate, mkt_clearing_simple], name="HA Model with cyclical income risk")

#Economy with MP - variable kappa 

@sj.simple
def nkpc(pi, Y, yss, T, C, Z,theta_w, vphi, frisch, markup_ss, eis, beta, cpshock, curve, epsilon, phi,dfshock):
    kappa_w_base = epsilon/phi
    kappa_w = kappa_w_base*(2.72**(curve*100*(Y-yss)))
    N = Y
    W = 1/markup_ss
    piw = (1+pi) * W / W(-1) -1
    piwres= kappa_w*N*(vphi * (N)**(1/frisch)-(epsilon-1)/epsilon*C**(-1/eis)*W*(1-T)) + beta*2.72**(dfshock)*piw(1)*(1+piw(1)) - piw*(1+piw) + cpshock
    return piwres, piw, kappa_w,N,W



@sj.simple
def monetary_taylor_v(pi, Y,yss,ishock, phi_pi, phi_y,rss): 
    i=((1+rss)*(pi+1)**phi_pi*  (Y/yss)**phi_y)*2.72** ishock -1
    r_ante = (1+i) / (1+pi(1)) -1
    return i, r_ante


# We'll also define an RA model for comparison

#In[7]:


@sj.solved(unknowns={'C': 1, 'A': 1}, targets=["euler", "budget_constraint"], solver="broyden_custom")
def household_ra_simple(C, A, Y, eis, beta, r):
    euler = (beta * (1 + r(1))) ** (-eis) * C(1) - C
    budget_constraint = (1 + r) * A(-1) + Y - C - A
    return euler, budget_constraint

ra = sj.create_model([household_ra_simple, ex_post_rate, mkt_clearing_simple], name="Representative Agent Model")


ss = {}
models = {'ha': ha_simple, 'ra': ra}

print('Model Set-up done')


# %% Cyclical HANK model setup

models['ha_taylor_cyc'] = sj.create_model([household_cyc, mkt_clearing_simple, nkpc,
                                           ex_post_rate, monetary_taylor_v, fiscal], name='Cyclical HA w/ Taylor Rule')

calib_taylor_cyc = calibration.copy()
calib_taylor_cyc.update({'X': 1., 'rss': calibration['r_ante'], 'pi': 0., 'ishock': 0.,
                         'phi_pi': 1.5, 'phi_y': 0.2, 'theta_w': 0.85, 'frisch': 0.5, 'markup_ss': 1.2, 'cpshock': 0, 'dfshock': 0, 'Tr': 0})
calib_taylor_cyc['zeta'] = -4
calib_taylor_cyc['zetap'] = -10
ss['ha_taylor_cyc'] = models['ha_taylor_cyc'].solve_steady_state(
    calib_taylor_cyc, {'beta': 0.99, 'vphi': 0.8}, ['asset_mkt', 'piwres'])


# %% Create storage
irf_i_ha, G_ha, irf_Y_ha, irf_r_post_ha, irf_pi_ha, irf_kappa_ha, irf_internals, irf_internals_D , irf_internals_a, irf_internals_y, irf_internals_p,irf_internals_pre,irf_internals_p_pre= {}, {}, {}, {},{},{},{},{},{},{},{},{},{}

# %% Cyclical

T_horizon = 100
T = 1000
rng = default_rng()
mu, sigma = 0, 0.01/22 #21.25
i_shock_list = np.random.default_rng(seed=123456).normal(mu, sigma, T)
mu_cp, sigma_cp = 0, 0.01/13.15
cp_shock_list = np.random.default_rng(seed=123457).normal(mu_cp, sigma_cp, T)


i_shock_test = [0]
di = i_shock_test * 0.9 ** np.arange(T_horizon)
piguess = di*0
Yguess = di*0
i_shock_old = 0
cp_shock_old = 0

elementstwointernals = []
elementstwointernals_y = []
elementstwointernals_p = []
elementstwointernals_p_pre = []
elementstwointernals_pre = []
elementstwointernals_a = []
elementstwointernals_D = []
elementstwointernals_Dbeg = []
calibration_theta = copy.deepcopy(calib_taylor_cyc)
ss_nom_ha = models['ha_taylor_cyc'].solve_steady_state(
    calibration_theta, {'beta': 0.99, 'vphi': 0.8}, ['asset_mkt', 'piwres'])
ss_in = deepcopy(ss_nom_ha)
outputs = ['Y', 'r_ante', 'pi', 'i', 'kappa_w']
td_total = {out: np.zeros(T + T_horizon)
            for out in outputs}  # store here the sum of all shocks
td_individual = {out: np.zeros((T, T + T_horizon))
                 for out in outputs}  # store here all individual shocks
td_total_internals = {}
td_total_internals_a = {}
td_total_internals_y = {}
td_total_internals_p = {}
td_total_internals_p_pre = {}
td_total_internals_pre = {}
td_total_internals_D = {}
td_total_internals_Dbeg = {}
td_individual_internals = {}
for i, i_shock in enumerate(i_shock_list):
    di = i_shock_old + i_shock * 0.9**np.arange(T_horizon)
    dcp = cp_shock_old + cp_shock_list[i] * 0.9**np.arange(T_horizon)
    calibration_theta = copy.deepcopy(calib_taylor_cyc)
    ss_nom_ha = models['ha_taylor_cyc'].solve_steady_state(
        calibration_theta, {'beta': 0.99, 'vphi': 0.8}, ['asset_mkt', 'piwres'])
    irf_here_ha = models['ha_taylor_cyc'].solve_impulse_nonlinear_LST_homotopy(ss_nom_ha, ['Y', 'pi'], ['asset_mkt', 'piwres'], {
                                                                           'dfshock': di, 'cpshock': dcp}, {'Yguess': Yguess}, {'piguess': piguess}, internals=['hh'], ss_initial=ss_in)   
    ss_in.update({k: ss_nom_ha[k] + irf_here_ha[k][0] for k in irf_here_ha})
    ss_in.internals['hh'].update({k: ss_nom_ha.internals['hh'][k] +
                                 irf_here_ha.internals['hh'][k][0, ...] for k in irf_here_ha.internals['hh']})
    i_shock_old = np.concatenate((di[1:], np.array([0])))
    cp_shock_old = np.concatenate((dcp[1:], np.array([0])))

    piguess = irf_here_ha['pi']
    Yguess = irf_here_ha['Y']

    for out in outputs:
        td_total[out][i:i+T_horizon] = irf_here_ha[out]
        td_individual[out][i, i:i+T_horizon] = irf_here_ha[out][np.newaxis, :]

    irf_internals[i] = irf_here_ha.internals['hh']['c']
    elementstwointernals.append(irf_internals[i][0])
    td_total_internals = np.array(elementstwointernals)

    irf_internals_y[i] = irf_here_ha.internals['hh']['z']
    elementstwointernals_y.append(irf_internals_y[i][0])
    td_total_internals_y = np.array(elementstwointernals_y)

    irf_internals_a[i] = irf_here_ha.internals['hh']['a']
    elementstwointernals_a.append(irf_internals_a[i][0])
    td_total_internals_a = np.array(elementstwointernals_a)

    irf_internals_D[i] = irf_here_ha.internals['hh']['D']
    elementstwointernals_D.append(irf_internals_D[i][0])
    td_total_internals_D = np.array(elementstwointernals_D)

    irf_internals_p[i] = irf_here_ha.internals['hh']['p']
    elementstwointernals_p.append(irf_internals_p[i][0])
    td_total_internals_p = np.array(elementstwointernals_p)
    
    irf_internals_pre[i] = irf_here_ha.internals['hh']['pretax']
    elementstwointernals_pre.append(irf_internals_pre[i][0])
    td_total_internals_pre = np.array(elementstwointernals_pre)
    
    irf_internals_p_pre[i] = irf_here_ha.internals['hh']['p_pretax']
    elementstwointernals_p_pre.append(irf_internals_p_pre[i][0])
    td_total_internals_p_pre = np.array(elementstwointernals_p_pre)


    print('Shock Number ' + str(i+1))


    # %%Inequality 

T = 1000
a_ss = ss_nom_ha.internals['hh']['a']
c_ss = ss_nom_ha.internals['hh']['c']
y_ss = np.transpose(np.tile(ss_nom_ha.internals['hh']['pretax'], (500, 1)))
p_ss = np.transpose(np.tile(ss_nom_ha.internals['hh']['p_pretax'], (500, 1)))
r_ss = ss_nom_ha['r_ante']
inc_ss = y_ss+p_ss+r_ss*a_ss

a = td_total_internals_a
p = td_total_internals_p_pre
y = td_total_internals_pre
c = td_total_internals
r = td_total['r_ante']

a_abs = a+a_ss
r_abs = r+r_ss
c_abs=c+c_ss

le = np.empty((T, 11, 500))
pinc = np.empty((T, 11, 500))
inc = np.empty((T, 11, 500))


for j in range(0, T):

    le[j, :, :] = np.transpose(np.tile(y[j, :], (500, 1)))+y_ss
    pinc[j, :, :] = np.transpose(np.tile(p[j, :], (500, 1)))+p_ss
    inc[j, :, :] = le[j, :, :] +pinc[j,:,:]+r_abs[j]*a_abs[j,:,:] 

D_ss = ss_nom_ha.internals['hh']['D']
d = td_total_internals_D
D_abs = d+D_ss

inc_vec = np.empty((5500, T))
le_vec = np.empty((5500, T))
D_vec = np.empty((5500, T))
inc_sort = np.empty((5500, T))
le_sort = np.empty((5500, T))
D_sort = np.empty((5500, T))
D_sort_le = np.empty((5500, T))
c_sort = np.empty((5500, T))


inc_vec_ss = inc_ss.flatten()
le_vec_ss = y_ss.flatten()
D_vec_ss = D_ss.flatten()
indices = np.argsort(inc_vec_ss)
indices_le = np.argsort(le_vec_ss)
inc_sort_ss = inc_vec_ss[indices]
le_sort_ss = le_vec_ss[indices_le]
D_sort_ss = D_vec_ss[indices]
D_sort_le_ss = D_vec_ss[indices_le]

for j in range(0, T):
    inc_vec[:, j] = inc[j, :, :].flatten()
    le_vec[:, j] = le[j, :, :].flatten()
    D_vec[:, j] = D_abs[j, :, :].flatten()
    indices = np.argsort(inc_vec[:, j])
    indices_le = np.argsort(le_vec[:, j])
    inc_sort[:, j] = inc_vec[:, j][indices]
    le_sort[:, j] = le_vec[:, j][indices_le]
    D_sort[:, j] = D_vec[:, j][indices]
    D_sort_le[:, j] = D_vec[:, j][indices_le]

weighted_std_inc = np.empty((T))
weighted_std_le = np.empty((T))
inc_sort_ss_log = np.log(inc_sort_ss)
le_sort_ss_log = np.log(le_sort_ss)
inc_sort_log = np.log(inc_sort)
inc_sort_log[np.isnan(inc_sort_log)]=np.nanmin(inc_sort_log)
le_sort_log = np.log(le_sort)
for j in range(0, T):
    weighted_std_inc[j] = np.sqrt(np.sum(D_sort[:, j] * (inc_sort_log[:, j] - np.average(
        inc_sort_log[:, j], weights=D_sort[:, j]))**2) / np.sum(D_sort[:, j]))
    weighted_std_le[j] = np.sqrt(np.sum(D_sort_le[:, j] * (le_sort_log[:, j] - np.average(
        le_sort_log[:, j], weights=D_sort_le[:, j]))**2) / np.sum(D_sort_le[:, j]))


skewsdlognonlin_inc = scipy.stats.skew(weighted_std_inc, axis=0, bias=True)
stdlognonlin_inc = np.std(weighted_std_inc)
ss_value_sdlog_inc = np.sqrt(np.sum(D_sort_ss * (inc_sort_ss_log - np.average(
    inc_sort_ss_log, weights=D_sort_ss))**2) / np.sum(D_sort_ss))

skewsdlognonlin_le = scipy.stats.skew(weighted_std_le, axis=0, bias=True)
stdlognonlin_le = np.std(weighted_std_le)
ss_value_sdlog_le = np.sqrt(np.sum(D_sort_le_ss * (le_sort_ss_log - np.average(
    le_sort_ss_log, weights=D_sort_le_ss))**2) / np.sum(D_sort_ss))

infl_ann, y_ann, y_growth_base, y_growth = {}, {}, {}, {}

infl_ann = td_total['pi']*400

y_ann = (td_total['Y']+1)
for i in range(1, T):
    y_growth_base[i] = (y_ann[i]-y_ann[i-1])/(y_ann[i-1])*400
    y_growth = np.array(list(y_growth_base.values()))

stdinf, skewinf, autocorrinf, stdy, skewy, autocorry, corrinfly = {}, {}, {}, {}, {}, {}, {}

stdinf = np.std(infl_ann)
skewinf = scipy.stats.skew(infl_ann[1:T], axis=0, bias=True)
autocorrinf = np.corrcoef(infl_ann[:-1], infl_ann[1:])[0, 1]


stdy = np.std(y_growth)
skewy = scipy.stats.skew(y_growth, axis=0, bias=True)
autocorry = np.corrcoef(y_growth[:-1], y_growth[1:])[0, 1]

corrinfly = np.corrcoef(infl_ann[1:T], y_growth)[0, 1]



# %% Constant Phillips Curve Model

# In[3]: 

# initialize
def hh_init(a_grid, z, r, eis,dfshock,p):
    coh = (1 + r) * a_grid[np.newaxis, :] + z[:, np.newaxis]  + p[:, np.newaxis] 
    Va = (1 + r) * (coh) ** (-1 / eis)
    return Va

# backward step
@sj.het(exogenous='Pi',  # <-- this means our transition matrix will be fed into the model as Pi (use this for forward iteration)
        policy='a',  # <-- this means our endogenous state variable is a, defined over grid a_grid (we use this to check convergence)
        backward='Va',  # <-- this means we're iterating over variable Va, whose future value is Va_p (solver needs to know this to iterate!)
        backward_init=hh_init)
def hh(Va_p, a_grid, z, r, beta, eis,dfshock,Y,p):
    uc_nextgrid = beta *2.72**(dfshock)* Va_p  # u'(c') on tomorrow's grid
    c_nextgrid = uc_nextgrid ** (-eis) # c' on tomorrow's grid
    coh = (1 + r) * a_grid[np.newaxis, :] + z[:, np.newaxis] + p[:, np.newaxis] # cash on hand on today's grid
    a = sj.interpolate.interpolate_y(c_nextgrid + a_grid, coh, a_grid)  # this plots (c_next + a', a') pairs and computes policy a' from interpolation on coh
    sj.misc.setmin(a, a_grid[0])  # impose borrowing constraint
    c = coh - a  # back out consumption
    Va = (1 + r) * (c) ** (-1 / eis)  # V'(a)
    return Va, a, c,coh
    

# forward iteration is done automatically!

def income_cyclical(Y,Z,P,T,e_grid, e_pdf, zeta,zetap,markup_ss,p_grid,p_pdf):
    N =Y
    gamma_N = e_grid ** (zeta * np.log(N)) / np.vdot(e_grid ** (1 + zeta * np.log(N)), e_pdf)
    z = Z * e_grid *gamma_N
    pretax = (Y/markup_ss) * e_grid *gamma_N
    #gamma_P = p_grid ** (zetap * np.log(P/((1-T)*(1-1/markup_ss)))) / np.vdot(p_grid ** (1 + zetap * np.log(P/((1-T)*(1-1/markup_ss)))), p_pdf)
    gamma_P = p_grid ** (zetap * np.log(Y)) / np.vdot(p_grid ** (1 + zetap * np.log(Y)), p_pdf)
    p = (P)* p_grid * gamma_P
    p_pretax = (Y*(1-1/markup_ss))* p_grid * gamma_P
    return z, p, pretax, p_pretax


def make_grids_pdf(rho_e, rho_p, sd_e, sd_p, n_e, min_a, max_a, n_a):
    e_grid, e_pdf, Pi = sj.grids.markov_rouwenhorst(rho_e, sd_e, n_e)
    p_grid, p_pdf, Pi = sj.grids.markov_rouwenhorst(rho_p, sd_p, n_e)
    a_grid = sj.grids.asset_grid(min_a, max_a, n_a)
    return e_grid, e_pdf, Pi, a_grid,  p_grid, p_pdf

household_cyc = hh.add_hetinputs([make_grids_pdf, income_cyclical])

@sj.simple
def ex_post_rate(r_ante):
    r = r_ante
    return r

@sj.simple
def fiscal(B, r, G, Y, Tr,dfshock,markup_ss):
    T = ((1 + r) * B(-1) + G - B + Tr)/(Y) # total tax burden
    Z = (Y/markup_ss)*(1 - T)  # after tax income
    P = Y*(1-1/markup_ss)*(1-T)
    deficit = G + Tr - T
    return T, Z, deficit,P

@sj.simple
def mkt_clearing_simple(A, B, Y, C, G):
    asset_mkt = A - B
    goods_mkt = Y - C - G 
    return asset_mkt, goods_mkt

ha_simple = sj.create_model([household_cyc, ex_post_rate, mkt_clearing_simple])
ha_cyc = sj.create_model([household_cyc, ex_post_rate, mkt_clearing_simple], name="HA Model with cyclical income risk")

#Economy with MP - variable kappa 

@sj.simple
def nkpc(pi, Y, yss, T, C, Z,theta_w, vphi, frisch, markup_ss, eis, beta, cpshock, curve, epsilon, phi,dfshock):
    kappa_w_base = epsilon/phi
    kappa_w = kappa_w_base*(2.72**(curve*100*(Y-yss)))
    N = Y
    W = 1/markup_ss
    piw = (1+pi) * W / W(-1) -1
    piwres= kappa_w*N*(vphi * (N)**(1/frisch)-(epsilon-1)/epsilon*C**(-1/eis)*W*(1-T)) + beta*2.72**(dfshock)*piw(1)*(1+piw(1)) - piw*(1+piw) + cpshock
    return piwres, piw, kappa_w,N,W



@sj.simple
def monetary_taylor_v(pi, Y,yss,ishock, phi_pi, phi_y,rss): 
    i=((1+rss)*(pi+1)**phi_pi*  (Y/yss)**phi_y)*2.72** ishock -1
    r_ante = (1+i) / (1+pi(1)) -1
    return i, r_ante


# We'll also define an RA model for comparison

#In[7]:


@sj.solved(unknowns={'C': 1, 'A': 1}, targets=["euler", "budget_constraint"], solver="broyden_custom")
def household_ra_simple(C, A, Y, eis, beta, r):
    euler = (beta * (1 + r(1))) ** (-eis) * C(1) - C
    budget_constraint = (1 + r) * A(-1) + Y - C - A
    return euler, budget_constraint

ra = sj.create_model([household_ra_simple, ex_post_rate, mkt_clearing_simple], name="Representative Agent Model")


ss = {}
models = {'ha': ha_simple, 'ra': ra}

print('Model Set-up done')

# In[7]:


@sj.solved(unknowns={'C': 1, 'A': 1}, targets=["euler", "budget_constraint"], solver="broyden_custom")
def household_ra_simple(C, A, Y, eis, beta, r):
    euler = (beta * (1 + r(1))) ** (-eis) * C(1) - C
    budget_constraint = (1 + r) * A(-1) + Y - C - A
    return euler, budget_constraint


ra = sj.create_model([household_ra_simple, ex_post_rate,
                     mkt_clearing_simple], name="Representative Agent Model")


ss = {}
models = {'ra': ra}

print('Model Set-up done')


# %% Cyclical HANK model setup

models['ha_taylor_cyc'] = sj.create_model([household_cyc, mkt_clearing_simple, nkpc,
                                           ex_post_rate, monetary_taylor_v, fiscal], name='Cyclical HA w/ Taylor Rule')

calib_taylor_cyc = calibration.copy()
calib_taylor_cyc.update({'X': 1., 'rss': calibration['r_ante'], 'pi': 0., 'ishock': 0.,
                         'curve':0,'phi_pi': 1.5, 'phi_y': 0.2, 'theta_w': 0.85, 'frisch': 0.5, 'markup_ss': 1.2, 'cpshock': 0,'dfshock': 0,'Tr': 0.})
calib_taylor_cyc['zeta'] = -4
calib_taylor_cyc['zetap'] = -10

ss['ha_taylor_cyc'] = models['ha_taylor_cyc'].solve_steady_state(
    calib_taylor_cyc, {'beta': 0.99, 'vphi': 0.8}, ['asset_mkt', 'piwres'])

# %% Long simulation 


T_horizon = 100
T = 1000
rng = default_rng()
mu, sigma = 0, 0.01/22
i_shock_list = np.random.default_rng(seed=123456).normal(mu, sigma, T)
mu_cp, sigma_cp = 0, 0.01/13.15 #.5 /9.8
cp_shock_list = np.random.default_rng(seed=123457).normal(mu_cp, sigma_cp, T)
i_shock_test = [0]
di = i_shock_test * 0.9 ** np.arange(T_horizon)
piguess = di*0
Yguess = di*0
phi_pi = 1.5

i_shock_old = 0
cp_shock_old = 0
td_total_lin_internals = {}
td_total_lin_internals_y = {}
td_total_lin_internals_a = {}
td_total_lin_internals_p = {}
td_total_lin_internals_D = {}
td_total_lin_internals_Dbeg = {}
td_total_lin_internals_p_pre = {}
td_total_lin_internals_pre = {}
td_individual_lin_internals = {}
elementstwointernals = []
elementstwointernals_a = []
elementstwointernals_p = []
elementstwointernals_pre = []
elementstwointernals_p_pre = []
elementstwointernals_y = []
elementstwointernals_D = []
elementstwointernals_Dbeg = []
ss_nom_ha = models['ha_taylor_cyc'].solve_steady_state(
    calib_taylor_cyc, {'beta': 0.99, 'vphi': 0.8}, ['asset_mkt', 'piwres'])
ss_in = deepcopy(ss_nom_ha)
outputs = ['Y', 'r_ante', 'pi', 'i', 'kappa_w']
# store here the sum of all shocks
td_total_lin = {out: np.zeros(T + T_horizon) for out in outputs}
td_individual_lin = {out: np.zeros((T, T + T_horizon)) for out in outputs}
for i, i_shock in enumerate(i_shock_list):
    di = i_shock_old + i_shock * 0.9**np.arange(T_horizon)
    dcp = cp_shock_old + cp_shock_list[i] * 0.9**np.arange(T_horizon)
    ss_nom_ha = models['ha_taylor_cyc'].solve_steady_state(
        calib_taylor_cyc, {'beta': 0.99, 'vphi': 0.8}, ['asset_mkt', 'piwres'])
    irf_here_ha_lin = models['ha_taylor_cyc'].solve_impulse_nonlinear_LST_homotopy(ss_nom_ha, ['Y', 'pi'], ['asset_mkt', 'piwres'], {
                                                                                   'ishock': di, 'cpshock': dcp}, {'Yguess': Yguess}, {'piguess': piguess}, internals=['hh'], ss_initial=ss_in)
    ss_in.update({k: ss_nom_ha[k] + irf_here_ha_lin[k][0]
                 for k in irf_here_ha_lin})
    ss_in.internals['hh'].update({k: ss_nom_ha.internals['hh'][k] +
                                 irf_here_ha_lin.internals['hh'][k][0, ...] for k in irf_here_ha_lin.internals['hh']})
    i_shock_old = np.concatenate((di[1:], np.array([0])))
    cp_shock_old = np.concatenate((dcp[1:], np.array([0])))

    piguess = irf_here_ha_lin['pi']
    Yguess = irf_here_ha_lin['Y']

    for out in outputs:
        td_total_lin[out][i:i+T_horizon] = irf_here_ha_lin[out]
        td_individual_lin[out][i, i:i +
                               T_horizon] = irf_here_ha_lin[out][np.newaxis, :]

    irf_internals[i] = irf_here_ha_lin.internals['hh']['c']
    elementstwointernals.append(irf_internals[i][0])
    td_total_lin_internals = np.array(elementstwointernals)

    irf_internals_y[i] = irf_here_ha_lin.internals['hh']['z']
    elementstwointernals_y.append(irf_internals_y[i][0])
    td_total_lin_internals_y = np.array(elementstwointernals_y)

    irf_internals_a[i] = irf_here_ha_lin.internals['hh']['a']
    elementstwointernals_a.append(irf_internals_a[i][0])
    td_total_lin_internals_a = np.array(elementstwointernals_a)
    
    irf_internals_p[i] = irf_here_ha_lin.internals['hh']['p']
    elementstwointernals_p.append(irf_internals_p[i][0])
    td_total_lin_internals_p = np.array(elementstwointernals_p)

    irf_internals_D[i] = irf_here_ha_lin.internals['hh']['D']
    elementstwointernals_D.append(irf_internals_D[i][0])
    td_total_lin_internals_D = np.array(elementstwointernals_D)

    irf_internals_pre[i] = irf_here_ha_lin.internals['hh']['pretax']
    elementstwointernals_pre.append(irf_internals_pre[i][0])
    td_total_lin_internals_pre = np.array(elementstwointernals_pre)
    
    irf_internals_p_pre[i] = irf_here_ha_lin.internals['hh']['p_pretax']
    elementstwointernals_p_pre.append(irf_internals_p_pre[i][0])
    td_total_lin_internals_p_pre = np.array(elementstwointernals_p_pre)
    
    print('Shock Number ' + str(i+1))  # %%

   # %%Inequality
T = 1000
a_ss = ss_nom_ha.internals['hh']['a']
c_ss = ss_nom_ha.internals['hh']['c']
y_ss = np.transpose(np.tile(ss_nom_ha.internals['hh']['pretax'], (500, 1)))
p_ss = np.transpose(np.tile(ss_nom_ha.internals['hh']['p_pretax'], (500, 1)))
r_ss = ss_nom_ha['r_ante']
inc_ss = y_ss   +p_ss+r_ss*a_ss

a = td_total_lin_internals_a
p = td_total_lin_internals_p_pre
y = td_total_lin_internals_pre
c = td_total_lin_internals
r = td_total_lin['r_ante']

a_abs = a+a_ss
r_abs = r+r_ss
c_abs=c+c_ss

le = np.empty((T, 11, 500))
pinc = np.empty((T, 11, 500))
inc = np.empty((T, 11, 500))


for j in range(0, T):

    le[j, :, :] = np.transpose(np.tile(y[j, :], (500, 1)))+y_ss
    pinc[j, :, :] = np.transpose(np.tile(p[j, :], (500, 1)))+p_ss
    inc[j, :, :] = le[j, :, :] +pinc[j,:,:]+r_abs[j]*a_abs[j,:,:] 

D_ss = ss_nom_ha.internals['hh']['D']
d = td_total_lin_internals_D
D_abs = d+D_ss

inc_vec = np.empty((5500, T))
le_vec = np.empty((5500, T))
D_vec = np.empty((5500, T))
inc_sort = np.empty((5500, T))
le_sort = np.empty((5500, T))
D_sort = np.empty((5500, T))
c_sort = np.empty((5500, T))


inc_vec_ss = inc_ss.flatten()
le_vec_ss = y_ss.flatten()
D_vec_ss = D_ss.flatten()
indices = np.argsort(inc_vec_ss)
inc_sort_ss = inc_vec_ss[indices]
le_sort_ss = le_vec_ss[indices]
D_sort_ss = D_vec_ss[indices]

for j in range(0, T):
    inc_vec[:, j] = inc[j, :, :].flatten()
    le_vec[:, j] = le[j, :, :].flatten()
    D_vec[:, j] = D_abs[j, :, :].flatten()
    indices = np.argsort(inc_vec[:, j])
    inc_sort[:, j] = inc_vec[:, j][indices]
    le_sort[:, j] = le_vec[:, j][indices]
    D_sort[:, j] = D_vec[:, j][indices]

weighted_std_inc_lin = np.empty((T))
weighted_std_le_lin = np.empty((T))
inc_sort_ss_log = np.log(inc_sort_ss)
le_sort_ss_log = np.log(le_sort_ss)
inc_sort_log = np.log(inc_sort)
inc_sort_log[np.isnan(inc_sort_log)]=np.nanmin(inc_sort_log)
le_sort_log = np.log(le_sort)
for j in range(0, T):
    weighted_std_inc_lin[j] = np.sqrt(np.sum(D_sort[:, j] * (inc_sort_log[:, j] - np.average(
        inc_sort_log[:, j], weights=D_sort[:, j]))**2) / np.sum(D_sort[:, j]))
    weighted_std_le_lin[j] = np.sqrt(np.sum(D_sort[:, j] * (le_sort_log[:, j] - np.average(
        le_sort_log[:, j], weights=D_sort[:, j]))**2) / np.sum(D_sort[:, j]))


skewsdloglin_inc = scipy.stats.skew(weighted_std_inc_lin, axis=0, bias=True)
stdloglin_inc = np.std(weighted_std_inc_lin)
ss_value_sdlog_inc = np.sqrt(np.sum(D_sort_ss * (inc_sort_ss_log - np.average(
    inc_sort_ss_log, weights=D_sort_ss))**2) / np.sum(D_sort_ss))

skewsdlin_le = scipy.stats.skew(weighted_std_le_lin, axis=0, bias=True)
stdloglin_le = np.std(weighted_std_le_lin)
ss_value_sdlog_le = np.sqrt(np.sum(D_sort_ss * (le_sort_ss_log - np.average(
    le_sort_ss_log, weights=D_sort_ss))**2) / np.sum(D_sort_ss))

infl_ann, y_ann, y_growth_base, y_growth = {}, {}, {}, {}

infl_ann = td_total_lin['pi']*400

y_ann = (td_total_lin['Y']+1)
for i in range(1, T):
    y_growth_base[i] = (y_ann[i]-y_ann[i-1])/(y_ann[i-1])*400
    y_growth = np.array(list(y_growth_base.values()))

stdinf_lin, skewinf_lin, autocorrinf_lin, stdy_lin, skewy_lin, autocorry_lin, corrinfly_lin = {}, {}, {}, {}, {}, {}, {}

stdinf_lin = np.std(infl_ann)
skewinf_lin = scipy.stats.skew(infl_ann[1:T], axis=0, bias=True)
autocorrinf_lin = np.corrcoef(infl_ann[:-1], infl_ann[1:])[0, 1]


stdy_lin = np.std(y_growth)
skewy_lin = scipy.stats.skew(y_growth, axis=0, bias=True)
autocorry_lin = np.corrcoef(y_growth[:-1], y_growth[1:])[0, 1]

corrinfly_lin = np.corrcoef(infl_ann[1:T], y_growth)[0, 1]


# %% Figure 4

import matplotlib.pyplot as plt
import numpy as np
from scipy.stats import skew

# Compute skewness for each dataset (Red = td_total, Blue = td_total_lin)
skewness_Y_red = skew(400 * ((1+td_total['Y'][1:T])-(1+td_total['Y'][0:T-1]))/(1+td_total['Y'][0:T-1]))
skewness_Y_blue = skew(400 *((1+td_total_lin['Y'][1:T])-(1+td_total_lin['Y'][0:T-1]))/(1+td_total_lin['Y'][0:T-1]))

skewness_pi_red = skew(400 * td_total['pi'][0:T])
skewness_pi_blue = skew(400 * td_total_lin['pi'][0:T])

skewness_std_le_red = skew(((weighted_std_le[0:T] - ss_value_sdlog_le) / ss_value_sdlog_le) * 100)
skewness_std_le_blue = skew(((weighted_std_le_lin[0:T] - ss_value_sdlog_le) / ss_value_sdlog_le) * 100)

skewness_std_inc_red = skew(((weighted_std_inc[0:T] - ss_value_sdlog_inc) / ss_value_sdlog_inc) * 100)
skewness_std_inc_blue = skew(((weighted_std_inc_lin[0:T] - ss_value_sdlog_inc) / ss_value_sdlog_inc) * 100)

# Create subplots
fig, ((ax1, ax2), (ax3, ax4)) = plt.subplots(2, 2, sharex=False, sharey=False, figsize=(18, 12))

# Function to add skewness text boxes for both red and blue graphs
def add_skewness_text(ax, skew_red, skew_blue):
    ax.text(0.05, 0.9, f'Skewness = {skew_red:.2f}', transform=ax.transAxes,
            fontsize=12, color='red', bbox=dict(facecolor='white', edgecolor='red', boxstyle='round,pad=0.3'))
    
    ax.text(0.05, 0.82, f'Skewness = {skew_blue:.2f}', transform=ax.transAxes,
            fontsize=12, color='blue', bbox=dict(facecolor='white', edgecolor='blue', boxstyle='round,pad=0.3'))


# ---- Subplot 1: Inflation ----  <<< Now Inflation first
ax1.plot((400 * td_total['pi'][0:T]), 'r-', alpha=0.9)
ax1.plot((400 * td_total_lin['pi'][0:T]), 'b--', alpha=0.6)
ax1.set_title('Inflation (annualized)')
ax1.set(ylabel='p.p. deviation from steady state')
ax1.axhline(y=0, color='#808080', linestyle=':')
ax1.spines['right'].set_visible(False)
ax1.spines['top'].set_visible(False)
add_skewness_text(ax1, skewness_pi_red, skewness_pi_blue)

# ---- Subplot 2: Output Growth ----  <<< Now Output second
h1, = ax2.plot((400 * ((1+td_total['Y'][1:T])-(1+td_total['Y'][0:T-1]))/(1+td_total['Y'][0:T-1])), 'r-', label="State-dependent Phillips curve slope" , alpha=0.9)
h2, = ax2.plot((400 *((1+td_total_lin['Y'][1:T])-(1+td_total_lin['Y'][0:T-1]))/(1+td_total_lin['Y'][0:T-1])), 'b--', label="Constant Phillips curve slope" , alpha=0.6)
ax2.set_title('Output Growth (annualized)')
ax2.set(ylabel='percent deviation from steady state')
ax2.axhline(y=0, color='#808080', linestyle=':')
ax2.spines['right'].set_visible(False)
ax2.spines['top'].set_visible(False)
add_skewness_text(ax2, skewness_Y_red, skewness_Y_blue)

# ---- Subplot 3: Standard deviation of log labor income ----
ax3.plot(((weighted_std_le[0:T] - ss_value_sdlog_le) / ss_value_sdlog_le) * 100, 'r-', alpha=0.9)
ax3.plot(((weighted_std_le_lin[0:T] - ss_value_sdlog_le) / ss_value_sdlog_le) * 100, 'b--' , alpha=0.6)
ax3.set_title('Labor Income Inequality')
ax3.set(xlabel='quarters', ylabel='percent deviation from steady state')
ax3.axhline(y=0, color='#808080', linestyle=':')
ax3.spines['right'].set_visible(False)
ax3.spines['top'].set_visible(False)
add_skewness_text(ax3, skewness_std_le_red, skewness_std_le_blue)

# ---- Subplot 4: Standard deviation of log labor and capital income ----
ax4.plot(((weighted_std_inc[0:T] - ss_value_sdlog_inc) / ss_value_sdlog_inc) * 100, 'r-', alpha=0.9)
ax4.plot(((weighted_std_inc_lin[0:T] - ss_value_sdlog_inc) / ss_value_sdlog_inc) * 100, 'b--', alpha=0.6)
ax4.set_title('Labor and Capital Income Inequality')
ax4.set(xlabel='quarters', ylabel='percent deviation from steady state')
ax4.axhline(y=0, color='#808080', linestyle=':')
ax4.spines['right'].set_visible(False)
ax4.spines['top'].set_visible(False)
add_skewness_text(ax4, skewness_std_inc_red, skewness_std_inc_blue)

# ---- Restore the Legend ----
fig.legend(handles=[h1, h2], labels=["State-dependent Phillips curve slope", "Constant Phillips curve slope"], 
           loc='lower center', ncol=2, fontsize=16, frameon=False)

plt.tight_layout(rect=[0, 0.05, 1, 1])  
plt.show()



